import Header from "@/components/Header";
import Footer from "@/components/Footer";
import { ChevronRight } from "lucide-react";
import { useEffect, useState, useRef } from "react";
import { useQuery } from "@tanstack/react-query";
import { renderSection } from "@/utils/renderSection";

interface UniversalCMSPageProps {
  pageId: string;
  pageTitle: string;
  pageSubtitle: string;
  breadcrumbText: string;
  breadcrumbPath?: { label: string; href?: string }[];
  floatingIcon?: React.ComponentType<any>;
}

export default function UniversalCMSPage({ 
  pageId, 
  pageTitle, 
  pageSubtitle, 
  breadcrumbText,
  breadcrumbPath,
  floatingIcon: FloatingIcon
}: UniversalCMSPageProps) {
  const [visibleSections, setVisibleSections] = useState<Set<number>>(new Set([0, 1])); // Make first 2 sections visible by default
  const [expandedTrustees, setExpandedTrustees] = useState<Set<string>>(new Set());
  const sectionRefs = useRef<(HTMLDivElement | null)[]>([]);

  const toggleTrusteeExpansion = (trusteeKey: string) => {
    setExpandedTrustees(prev => {
      const newSet = new Set(prev);
      if (newSet.has(trusteeKey)) {
        newSet.delete(trusteeKey);
      } else {
        newSet.add(trusteeKey);
      }
      return newSet;
    });
  };

  // Fetch sections from database using the provided page ID
  const { data: sections = [], isLoading: sectionsLoading, error } = useQuery({
    queryKey: ["backend/api/cms/pages", pageId, "sections"],
    queryFn: async () => {
      const response = await fetch(`backend/api/cms/pages/${pageId}/sections`);
      if (!response.ok) {
        throw new Error(`Failed to fetch sections: ${response.status}`);
      }
      return response.json();
    },
    retry: 1, // Retry once on failure
    refetchOnWindowFocus: false, // Don't refetch when window gains focus
  });

  useEffect(() => {
    const observer = new IntersectionObserver(
      (entries) => {
        entries.forEach((entry) => {
          const sectionIndex = parseInt(entry.target.getAttribute('data-section') || '0');
          if (entry.isIntersecting) {
            setVisibleSections(prev => new Set(Array.from(prev).concat(sectionIndex)));
          }
        });
      },
      {
        threshold: 0.2,
        rootMargin: '-50px'
      }
    );

    sectionRefs.current.forEach((ref) => {
      if (ref) observer.observe(ref);
    });

    return () => observer.disconnect();
  }, [sections.length]); // Re-run when sections load from database

  return (
    <>
      <Header />
      <div className="min-h-screen bg-gradient-to-br from-slate-50 via-blue-50/30 to-indigo-50 dark:from-slate-900 dark:to-slate-800">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20">
          {/* Breadcrumbs */}
          <div className="flex items-center text-sm text-gray-600 dark:text-gray-400 mb-8 animate-in slide-in-from-left duration-500">
            {breadcrumbPath ? (
              <>
                {breadcrumbPath.map((item, index) => (
                  <div key={index} className="flex items-center">
                    {index > 0 && <ChevronRight className="h-4 w-4 mx-2 text-gray-400" />}
                    {item.href ? (
                      <a href={item.href} className="hover:text-[#ED1C24] transition-colors">{item.label}</a>
                    ) : (
                      <span>{item.label}</span>
                    )}
                  </div>
                ))}
                <ChevronRight className="h-4 w-4 mx-2 text-gray-400" />
                <span className="text-[#ED1C24] font-medium">{breadcrumbText}</span>
              </>
            ) : (
              <>
                <span>Who We Are?</span>
                <ChevronRight className="h-4 w-4 mx-2 text-gray-400" />
                <span className="text-[#ED1C24] font-medium">{breadcrumbText}</span>
              </>
            )}
          </div>

          {/* Page Header */}
          <div className="mb-16 animation-delay-200"
               style={{
                 animation: 'slideInFromBottom 0.8s ease-out forwards'
               }}>
            <h1 className="text-5xl font-bold text-[#1A1A40] dark:text-white mb-6 leading-tight">
              {pageTitle}
            </h1>
            <div className="border-l-4 border-[#ED1C24] pl-6 bg-gradient-to-r from-[#ED1C24]/5 to-transparent p-4 rounded-r-lg">
              <p className="text-xl text-gray-600 dark:text-gray-400 font-medium">
                {pageSubtitle}
              </p>
            </div>
          </div>

          <div className="space-y-16">
            {/* Optional floating icon */}
            {FloatingIcon && (
              <div className="absolute top-40 right-10 opacity-10 dark:opacity-5 pointer-events-none">
                <div className="animate-float-slow">
                  <FloatingIcon className="h-32 w-32 text-[#ED1C24]" />
                </div>
              </div>
            )}

            {/* Dynamic Sections from Database using shared renderSection utility */}
            {sectionsLoading ? (
              <div className="text-center py-8">
                <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-[#ED1C24] mx-auto mb-4"></div>
                <p className="text-gray-600 dark:text-gray-400">Loading content...</p>
              </div>
            ) : error ? (
              <div className="text-center py-8">
                <p className="text-gray-600 dark:text-gray-400 mb-4">Unable to load dynamic content.</p>
                <p className="text-sm text-gray-500">Error: {error instanceof Error ? error.message : 'Unknown error'}</p>
              </div>
            ) : sections.length === 0 ? (
              <div className="bg-white dark:bg-slate-800 rounded-lg shadow-lg p-8">
                <div className="text-center">
                  <div className="w-20 h-20 bg-[#ED1C24] rounded-full flex items-center justify-center mx-auto mb-6">
                    <span className="text-2xl font-bold text-white">IBDL</span>
                  </div>
                  <h2 className="text-2xl font-semibold text-[#1A1A40] dark:text-white mb-4">
                    {pageTitle} Content Coming Soon
                  </h2>
                  <p className="text-gray-600 dark:text-gray-300 mb-6">
                    This page will feature comprehensive information about the {pageTitle} qualification, including curriculum details, certification requirements, and enrollment options.
                  </p>
                </div>
              </div>
            ) : (
              sections.map((section: any, index: number) => 
                renderSection(
                  section, 
                  index, 
                  visibleSections, 
                  expandedTrustees, 
                  toggleTrusteeExpansion, 
                  sectionRefs
                )
              )
            )}
          </div>
        </div>
      </div>
      <Footer />
    </>
  );
}