interface TemplateSelectionPanelProps {
  isOpen: boolean;
  selectedTemplate: string | null;
  templateConfig: Record<string, any>;
  onSelectTemplate: (templateKey: string) => void;
  onConfirm: () => void;
  onCancel: () => void;
  confirmLabel?: string;
  closeButtonTestId?: string;
  confirmButtonTestId?: string;
}

import { Button } from "@/components/ui/button";
import { Card, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";

export function TemplateSelectionPanel({
  isOpen,
  selectedTemplate,
  templateConfig,
  onSelectTemplate,
  onConfirm,
  onCancel,
  confirmLabel = "Continue with Template",
  closeButtonTestId = "button-close-template-selection",
  confirmButtonTestId = "button-confirm-template"
}: TemplateSelectionPanelProps) {
  if (!isOpen) return null;

  return (
    <div className="border rounded-lg p-6 bg-gray-50">
      <div className="flex items-center justify-between mb-4">
        <h3 className="text-lg font-semibold text-gray-900">Choose a Template</h3>
        <Button 
          variant="ghost" 
          size="sm" 
          onClick={onCancel}
          data-testid={closeButtonTestId}
        >
          ✕
        </Button>
      </div>
      <p className="text-gray-600 mb-6">Select a design template for your new section</p>
      
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {Object.entries(templateConfig).map(([templateKey, config]) => {
          const IconComponent = config.icon;
          const isSelected = selectedTemplate === templateKey;
          
          return (
            <Card 
              key={templateKey}
              className={`cursor-pointer transition-all duration-200 hover:shadow-md ${
                isSelected ? 'ring-2 ring-[#ED1C24] bg-red-50' : 'hover:bg-gray-50'
              }`}
              onClick={() => onSelectTemplate(templateKey)}
              data-testid={`template-${templateKey}`}
            >
              <CardHeader className="text-center p-4">
                <div className={`w-16 h-16 mx-auto rounded-lg flex items-center justify-center mb-3 ${
                  isSelected ? 'bg-[#ED1C24] text-white' : 'bg-gray-100 text-gray-600'
                }`}>
                  <IconComponent className="h-8 w-8" />
                </div>
                <CardTitle className="text-lg">{config.name}</CardTitle>
                <CardDescription className="text-sm text-gray-500">
                  {config.description}
                </CardDescription>
                <div className="mt-2 p-2 bg-white rounded border text-xs text-gray-600">
                  {config.preview}
                </div>
              </CardHeader>
            </Card>
          );
        })}
      </div>

      <div className="flex justify-end space-x-3 mt-6">
        <Button 
          variant="outline" 
          onClick={onCancel}
          data-testid="button-cancel-template"
        >
          Cancel
        </Button>
        <Button 
          className="bg-[#ED1C24] hover:bg-red-600 text-white"
          disabled={!selectedTemplate}
          onClick={onConfirm}
          data-testid={confirmButtonTestId}
        >
          {confirmLabel}
        </Button>
      </div>
    </div>
  );
}