import React, { useState } from "react";
import { Button } from "@/components/ui/button";
import { 
  ShoppingBag, Mail, Briefcase, Languages, ChevronDown, Menu, X, 
  UserPlus, GraduationCap, Book, BookOpen, Shield, Brain, Award, 
  Users, Building, Settings, Target, Lightbulb, TrendingUp,
  BookmarkPlus, Rocket, MonitorSpeaker, FileText, Handshake, 
  Headphones, Newspaper, Calendar, Globe, Star, CheckCircle, ArrowRight, Zap,
  BarChart3, Clock, LogOut, User
} from "lucide-react";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { useAuth } from "@/hooks/useAuth";
import { useLocation } from "wouter";
import ibdlLogo from "@assets/logo_1751230646761.png";

export default function Header() {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [expandedMobileSection, setExpandedMobileSection] = useState<string | null>(null);
  const { user, isAuthenticated, isLoading } = useAuth();
  const [location] = useLocation();

  // Helper function to check if current page is active
  const isActiveRoute = (href: string, section?: string) => {
    // First check for exact route match (most specific)
    if (location === href) return true;
    
    // Then check for section-level matches (less specific, only for navigation highlights)
    if (section === 'whoWeAre' && location.startsWith('/who-we-are')) return true;
    if (section === 'products' && location.startsWith('/our-products')) return true;
    if (section === 'services' && location.startsWith('/services')) return true;
    if (section === 'resources' && location.startsWith('/resources')) return true;
    
    return false;
  };

  const handleSignIn = () => {
    window.location.href = "/login";
  };

  const handleGetStarted = () => {
    window.location.href = "/get-started";
  };

  const handleMobileMenuToggle = () => {
    setMobileMenuOpen(!mobileMenuOpen);
    if (!mobileMenuOpen) {
      setExpandedMobileSection(null);
    }
  };

  // Hierarchical mega menu structure based on IBDL sitemap
  const megaMenus = {
    whoWeAre: {
      title: "Who We Are?",
      sections: [
        {
          title: "Our Story",
          icon: Book,
          href: "/who-we-are/our-story"
        },
        {
          title: "Leadership & Governance", 
          icon: Users,
          href: "/who-we-are/leadership-governance"
        },
        {
          title: "Our Impact",
          icon: TrendingUp,
          href: "/who-we-are/our-impact"
        },
        {
          title: "Knowledge Framework",
          icon: Brain,
          href: "/who-we-are/knowledge-framework"
        },
        {
          title: "Partners in Excellence",
          icon: Handshake,
          href: "/who-we-are/partners-in-excellence"
        }
      ]
    },
    products: {
      title: "Our Products",
      sections: [
        {
          title: "Qualifications",
          icon: GraduationCap,
          href: "/products/qualifications",
          subLinks: [
            { 
              name: "Core Qualifications", 
              items: [
                { name: "IBDL Essential", href: "/our-products/qualifications/core-qualifications/ibdl-essential" },
                { name: "IBDL Professional", href: "/our-products/qualifications/core-qualifications/ibdl-professional" },
                { name: "IBDL Master", href: "/our-products/qualifications/core-qualifications/ibdl-master" }
              ]
            },
            { 
              name: "Specialized Qualifications", 
              items: [
                { name: "Certified Educator", href: "/our-products/qualifications/specialized-qualifications/certified-educator" },
                { name: "Certified Industrial Engineer", href: "/our-products/qualifications/specialized-qualifications/certified-industrial-engineer" },
                { name: "CNPO Practitioner", href: "/our-products/qualifications/specialized-qualifications/cnpo-practitioner" },
                { name: "CNPO Management Professional", href: "/our-products/qualifications/specialized-qualifications/cnpo-management-professional" }
              ]
            },
            { 
              name: "Foresight Qualifications", 
              items: [
                { name: "Foresight & Innovation Associate", href: "/our-products/qualifications/foresight-qualifications/foresight-innovation-associate" },
                { name: "Foresight & Innovation Practitioner", href: "/our-products/qualifications/foresight-qualifications/foresight-innovation-practitioner" },
                { name: "Foresight & Innovation Strategist", href: "/our-products/qualifications/foresight-qualifications/foresight-innovation-strategist" }
              ]
            }
          ]
        },
        {
          title: "Learning Games",
          icon: Target,
          href: "/products/learning-games",
          subLinks: [
            { 
              name: "Core Games", 
              items: [
                { name: "Win vs. War", href: "/products/learning-games/win-vs-war" },
                { name: "Master Board Game", href: "/products/learning-games/master-board-game" },
                { name: "Sparta", href: "/products/learning-games/sparta" }
              ]
            },
            { 
              name: "Online Games", 
              items: [
                { name: "Mogul CEO", href: "/products/learning-games/mogul-ceo" },
                { name: "Maven Marketing", href: "/products/learning-games/maven-marketing" },
                { name: "Micromatic", href: "/products/learning-games/micromatic" }
              ]
            },
            { 
              name: "Specialized Games", 
              items: [
                { name: "Pitch Deck", href: "/products/learning-games/pitch-deck" },
                { name: "Synergy Stack", href: "/products/learning-games/synergy-stack" }
              ]
            }
          ]
        },
        {
          title: "IBDL Venture Lab",
          icon: Lightbulb,
          href: "/products/venture-hub",
          subLinks: [
            { 
              name: "Core Qualifications", 
              items: [
                { name: "Certified Disciplined Entrepreneur", href: "/products/venture-hub/certified-disciplined-entrepreneur" },
                { name: "Certified Incubation Management (CIM)", href: "/products/venture-hub/certified-incubation-management" }
              ]
            },
            { 
              name: "Specialized Executive Programs", 
              items: [
                { name: "Angel Playbook Bootcamp", href: "/products/venture-hub/angel-playbook-bootcamp" },
                { name: "Capital Raise Bootcamp", href: "/products/venture-hub/capital-raise-bootcamp" },
                { name: "Data-Room Preparation Bootcamp", href: "/products/venture-hub/data-room-preparation-bootcamp" }
              ]
            }
          ]
        },
        {
          title: "Assessment Services",
          icon: CheckCircle,
          href: "/products/assessment-services",
          subLinks: [
            { 
              name: "Core Assessment", 
              items: [
                { name: "PQP", href: "/products/assessment-services/pqp" },
                { name: "CPAT", href: "/products/assessment-services/cpat" }
              ]
            }
          ]
        },
        {
          title: "Executive Education",
          icon: Briefcase,
          href: "/products/executive-education"
        }
      ]
    },
    services: {
      title: "Our Services",
      sections: [
        {
          title: "Accreditation Services",
          icon: Shield,
          href: "/services/accreditation-services",
          subLinks: [
            { 
              name: "Provider Types", 
              items: [
                { name: "Training Provider", href: "/services/accreditation-services/training-provider" },
                { name: "Exam Provider", href: "/services/accreditation-services/exam-provider" },
                { name: "Corporate/Government", href: "/services/accreditation-services/corporate-government" }
              ]
            },
            { 
              name: "Professional Services", 
              items: [
                { name: "Professional IBDL Trainer", href: "/services/accreditation-services/professional-trainer" },
                { name: "Professional Course", href: "/services/accreditation-services/professional-course" }
              ]
            }
          ]
        },
        {
          title: "Be Part of Our Story",
          icon: Handshake,
          href: "/services/be-part-of-our-story",
          subLinks: [
            { 
              name: "Partnership", 
              items: [
                { name: "Partner With Us", href: "/services/be-part-of-our-story/partner-with-us" },
                { name: "Join Our Global Community", href: "/services/be-part-of-our-story/join-global-community" }
              ]
            }
          ]
        }
      ]
    },
    resources: {
      title: "Insights & Resources",
      sections: [
        {
          title: "Newsroom",
          icon: Newspaper,
          href: "/resources/newsroom"
        },
        {
          title: "The Learning Edit",
          icon: BookOpen,
          href: "/resources/learning-edit"
        },
        {
          title: "Program Brochures & Media Kit",
          icon: FileText,
          href: "/resources/program-brochures"
        },
        {
          title: "Events & Conferences",
          icon: Calendar,
          href: "/resources/events-conferences"
        }
      ]
    },
    contact: {
      title: "Contact Us",
      isSimple: true,
      href: "/contact"
    }
  };

  return (
    <div>
      {/* Navy Blue Top Bar */}
      <div className="bg-[#1A1A40] text-white">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-10 text-sm">
            <div className="flex items-center space-x-6">
              <span className="hidden md:block">Welcome to IBDL - Developing Learning Journeys Since 2006</span>
            </div>
            <div className="flex items-center space-x-4 md:space-x-6">
              <a href="/store" className="hover:text-[#ED1C24] transition-colors flex items-center">
                <ShoppingBag className="h-4 w-4 mr-1" />
                <span className="hidden sm:inline">Store</span>
              </a>
              <a href="/contact" className="hover:text-[#ED1C24] transition-colors flex items-center">
                <Mail className="h-4 w-4 mr-1" />
                <span className="hidden sm:inline">Contact Us</span>
              </a>
              <a href="#" className="hover:text-[#ED1C24] transition-colors flex items-center">
                <Briefcase className="h-4 w-4 mr-1" />
                <span className="hidden sm:inline">Careers</span>
              </a>
              <div className="relative group">
                <button className="flex items-center hover:text-[#ED1C24] transition-colors">
                  <Languages className="h-4 w-4 mr-1" />
                  <span>EN</span>
                  <ChevronDown className="h-3 w-3 ml-1" />
                </button>
                <div className="absolute right-0 top-full mt-1 bg-white text-gray-800 rounded-md shadow-lg py-1 min-w-[80px] opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-200 z-[9999]">
                  <a href="#" className="block px-3 py-1 hover:bg-gray-100 text-sm">English</a>
                  <a href="#" className="block px-3 py-1 hover:bg-gray-100 text-sm">العربية</a>
                  <a href="#" className="block px-3 py-1 hover:bg-gray-100 text-sm">Français</a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Main Navigation */}
      <nav className="navigation-sticky bg-white border-b border-gray-200 shadow-sm">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
          <div className="flex justify-between items-center h-16">
            <div className="flex items-center">
              <button 
                onClick={() => window.location.href = "/"}
                className="flex items-center hover:opacity-80 transition-opacity"
              >
                <img src={ibdlLogo} alt="IBDL Logo" className="h-10 w-auto" />
              </button>
            </div>
            
            {/* Desktop Navigation */}
            <div className="hidden xl:flex items-center space-x-8">
              {Object.entries(megaMenus).map(([key, menu]) => {
                const isActive = isActiveRoute('', key);
                
                // Handle simple links (like Contact Us)
                if ('isSimple' in menu && menu.isSimple) {
                  const isContactActive = isActiveRoute(menu.href!);
                  return (
                    <a 
                      key={key}
                      href={menu.href}
                      className={`relative px-3 py-2 text-sm transition-all duration-300 hover:text-[#ED1C24] ${
                        isContactActive 
                          ? 'text-[#ED1C24] font-bold after:content-[""] after:absolute after:bottom-0 after:left-0 after:w-full after:h-0.5 after:bg-[#ED1C24] after:rounded-full' 
                          : 'text-gray-700 font-medium'
                      }`}
                    >
                      <span className="relative z-10">{menu.title}</span>
                    </a>
                  );
                }

                // Handle complex mega menus
                return (
                  <div key={key} className="relative group">
                    <button className={`flex items-center px-3 py-2 text-sm transition-all duration-300 hover:text-[#ED1C24] group-hover:scale-105 whitespace-nowrap ${
                      isActive 
                        ? 'text-[#ED1C24] font-bold after:content-[""] after:absolute after:bottom-0 after:left-3 after:right-3 after:h-0.5 after:bg-[#ED1C24] after:rounded-full after:animate-pulse' 
                        : 'text-gray-700 font-medium'
                    }`}>
                      <span className="relative z-10">{menu.title}</span>
                      <ChevronDown className="ml-1 h-4 w-4 transition-transform duration-300 group-hover:rotate-180" />
                    </button>
                    
                    {/* Enhanced Mega Menu - Responsive Design */}
                    <div 
                      className={`mega-menu-dropdown absolute top-full mt-0 border-t-4 border-[#ED1C24] opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300 z-[90] rounded-b-lg overflow-hidden shadow-2xl animate-in slide-in-from-top-2 duration-300 ${
                        key === 'products' 
                          ? 'xl:left-0 xl:transform right-0 lg:right-0 md:right-0 sm:right-0' 
                          : 'left-1/2 transform -translate-x-1/2'
                      } max-w-[90vw] xl:max-w-none xl:w-fit`}
                      style={key === 'products' ? { 
                        '--tw-translate-x': '-26%',
                        transform: 'translate(var(--tw-translate-x), var(--tw-translate-y)) rotate(var(--tw-rotate)) skewX(var(--tw-skew-x)) skewY(var(--tw-skew-y)) scaleX(var(--tw-scale-x)) scaleY(var(--tw-scale-y))'
                      } as React.CSSProperties : {}}
                    >
                      {'sections' in menu && (
                        <>
                          {/* Single-layer sections (Who We Are, Insights & Resources) - Vertical Layout */}
                          {(key === 'whoWeAre' || key === 'resources') ? (
                            <div className="mega-menu-section p-4 xl:p-6 min-w-[240px] xl:min-w-[280px] max-w-[280px] xl:max-w-[320px]">
                              {/* Non-clickable Menu Title */}
                              <div className="mb-4 pb-3 border-b border-gray-100">
                                <h3 className="font-semibold text-[#1A1A40] text-lg">{menu.title}</h3>
                              </div>
                              
                              <div className="space-y-2">
                                {menu.sections.map((section: any, index: number) => (
                                  <a 
                                    key={index}
                                    href={section.href}
                                    className={`flex items-center py-1.5 px-2 rounded transition-all duration-200 hover:translate-x-2 hover:font-medium ${
                                      isActiveRoute(section.href) 
                                        ? 'text-[#ED1C24] font-medium' 
                                        : 'text-gray-600 hover:text-[#ED1C24]'
                                    }`}
                                  >
                                    {section.icon && <section.icon className={`h-4 w-4 mr-3 flex-shrink-0 transition-all duration-200 ${
                                      isActiveRoute(section.href) ? 'text-[#ED1C24]' : 'text-[#ED1C24]'
                                    }`} />}
                                    <span className="text-sm transition-all duration-200">
                                      {section.title}
                                    </span>
                                  </a>
                                ))}
                              </div>
                            </div>
                          ) : (
                            /* Multi-layer sections (Products, Services) - Responsive Horizontal Layout */
                            <div className="flex flex-wrap xl:flex-nowrap gap-0 w-fit max-w-[90vw] xl:max-w-none overflow-x-auto">
                              {menu.sections.map((section: any, index: number) => (
                                <div key={index} className="mega-menu-section p-4 xl:p-6 border-r border-gray-100 last:border-r-0 min-w-[200px] xl:min-w-fit flex-shrink-0">
                                  {/* Section Header with Icon - Clickable if no subLinks */}
                                  <div className="mb-3 xl:mb-4 pb-2 xl:pb-3 border-b border-gray-100">
                                    {section.subLinks ? (
                                      <div className="flex items-center whitespace-nowrap">
                                        {section.icon && <section.icon className="h-4 xl:h-5 w-4 xl:w-5 text-[#ED1C24] mr-2 xl:mr-3 flex-shrink-0" />}
                                        <h3 className="font-semibold text-[#1A1A40] text-base xl:text-lg">{section.title}</h3>
                                      </div>
                                    ) : (
                                      <a 
                                        href={section.href}
                                        className={`flex items-center whitespace-nowrap transition-colors duration-200 group ${
                                          isActiveRoute(section.href) 
                                            ? 'text-[#ED1C24]' 
                                            : 'hover:text-[#ED1C24]'
                                        }`}
                                      >
                                        {section.icon && <section.icon className={`h-4 xl:h-5 w-4 xl:w-5 mr-2 xl:mr-3 flex-shrink-0 transition-colors duration-200 ${
                                          isActiveRoute(section.href) 
                                            ? 'text-[#ED1C24]' 
                                            : 'text-gray-600 group-hover:text-[#ED1C24]'
                                        }`} />}
                                        <h3 
                                          className={`font-semibold text-base xl:text-lg transition-colors duration-200 ${
                                            isActiveRoute(section.href) 
                                              ? 'text-[#ED1C24]' 
                                              : 'text-[#1A1A40] group-hover:text-[#ED1C24]'
                                          }`}
                                          style={{
                                            color: isActiveRoute(section.href) ? '#ed1c24' : '#1a1a40'
                                          }}
                                        >{section.title}</h3>
                                      </a>
                                    )}
                                  </div>
                                  
                                  {/* Sub-sections - Clean List Style */}
                                  {section.subLinks && (
                                    <div className="space-y-3 xl:space-y-4">
                                      {section.subLinks.map((subLink: any, subIndex: number) => (
                                        <div key={subIndex}>
                                          <h4 className="font-medium text-gray-800 text-xs xl:text-sm mb-1 xl:mb-2 whitespace-nowrap">
                                            {subLink.name}
                                          </h4>
                                          <ul className="space-y-1 xl:space-y-1.5">
                                            {subLink.items.map((item: any, itemIndex: number) => (
                                              <li key={itemIndex}>
                                                <a 
                                                  href={typeof item === 'string' ? "#" : item.href} 
                                                  className={`text-xs py-1 xl:py-1.5 px-1 xl:px-2 block rounded transition-all duration-200 hover:text-[#ED1C24] hover:translate-x-1 xl:hover:translate-x-2 hover:font-medium whitespace-nowrap ${
                                                    typeof item !== 'string' && isActiveRoute(item.href) ? 'text-[#ED1C24] font-medium' : 'text-gray-600'
                                                  }`}
                                                >
                                                  {typeof item === 'string' ? item : item.name}
                                                </a>
                                              </li>
                                            ))}
                                          </ul>
                                        </div>
                                      ))}
                                    </div>
                                  )}
                                </div>
                              ))}
                            </div>
                          )}
                        </>
                      )}
                    </div>
                  </div>
                );
              })}
              
              <div className="flex space-x-3">
                {isAuthenticated && user ? (
                  <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                      <Button variant="ghost" className="flex items-center space-x-2 text-[#1A1A40] hover:text-[#ED1C24]">
                        <Avatar className="h-8 w-8">
                          <AvatarImage src={(user as any).profileImageUrl || undefined} />
                          <AvatarFallback className="bg-[#ED1C24] text-white text-sm">
                            {(user as any).firstName?.[0]?.toUpperCase() || (user as any).username?.[0]?.toUpperCase() || 'U'}
                          </AvatarFallback>
                        </Avatar>
                        <span className="hidden md:block">{(user as any).email}</span>
                        <ChevronDown className="h-4 w-4" />
                      </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent className="w-56" align="end">
                      <DropdownMenuLabel>My Account</DropdownMenuLabel>
                      <DropdownMenuSeparator />
                      <DropdownMenuItem>
                        <User className="mr-2 h-4 w-4" />
                        <span>Profile</span>
                      </DropdownMenuItem>
                      <DropdownMenuItem>
                        <Settings className="mr-2 h-4 w-4" />
                        <span>Settings</span>
                      </DropdownMenuItem>
                      <DropdownMenuSeparator />
                      <DropdownMenuItem onClick={() => window.location.href = "backend/api/logout"}>
                        <LogOut className="mr-2 h-4 w-4" />
                        <span>Logout</span>
                      </DropdownMenuItem>
                    </DropdownMenuContent>
                  </DropdownMenu>
                ) : (
                  <>
                    <Button 
                      onClick={handleGetStarted}
                      variant="outline"
                      className="border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white px-6 py-2 font-bold"
                    >
                      <UserPlus className="mr-2 h-4 w-4" />
                      Get Started
                    </Button>
                    <Button 
                      onClick={handleSignIn}
                      className="bg-[#ED1C24] hover:bg-red-700 text-white px-6 py-2 font-bold"
                    >
                      Sign In
                    </Button>
                  </>
                )}
              </div>
            </div>

            {/* Mobile menu button */}
            <div className="xl:hidden">
              <button
                onClick={handleMobileMenuToggle}
                className="text-gray-700 hover:text-[#ED1C24] transition-colors"
              >
                {mobileMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
              </button>
            </div>
          </div>
        </div>

        {/* Mobile Navigation */}
        {mobileMenuOpen && (
          <div className="xl:hidden bg-white border-t">
            <div className="px-4 py-6 space-y-6 max-h-[80vh] overflow-y-auto mobile-menu-scroll">
              {/* Action Buttons */}
              <div className="space-y-3 pb-4 border-b border-gray-200">
                {isAuthenticated && user ? (
                  <div className="space-y-3">
                    <div className="flex items-center space-x-3 p-3 bg-gray-50 rounded-lg">
                      <Avatar className="h-10 w-10">
                        <AvatarImage src={(user as any).profileImageUrl || undefined} />
                        <AvatarFallback className="bg-[#ED1C24] text-white">
                          {(user as any).firstName?.[0]?.toUpperCase() || (user as any).username?.[0]?.toUpperCase() || 'U'}
                        </AvatarFallback>
                      </Avatar>
                      <div className="flex-1">
                        <p className="font-medium text-[#1A1A40]">{(user as any).firstName || 'User'}</p>
                        <p className="text-sm text-gray-600">{(user as any).email}</p>
                      </div>
                    </div>
                    <Button 
                      onClick={() => {
                        window.location.href = "backend/api/logout";
                        setMobileMenuOpen(false);
                      }}
                      variant="outline"
                      className="w-full border-[#ED1C24] text-[#ED1C24] hover:bg-[#ED1C24] hover:text-white py-3"
                    >
                      <LogOut className="mr-2 h-4 w-4" />
                      Logout
                    </Button>
                  </div>
                ) : (
                  <>
                    <Button 
                      onClick={() => {
                        handleSignIn();
                        setMobileMenuOpen(false);
                      }}
                      className="w-full bg-[#ED1C24] hover:bg-red-700 text-white py-3 font-bold"
                    >
                      <UserPlus className="mr-2 h-4 w-4" />
                      Sign In
                    </Button>
                    <Button 
                      onClick={() => {
                        handleGetStarted();
                        setMobileMenuOpen(false);
                      }}
                      variant="outline"
                      className="w-full border-[#1A1A40] text-[#1A1A40] hover:bg-[#1A1A40] hover:text-white py-3 font-bold"
                    >
                      <UserPlus className="mr-2 h-4 w-4" />
                      Get Started
                    </Button>
                  </>
                )}
              </div>

              {/* Navigation Sections */}
              {Object.entries(megaMenus).map(([key, menu]) => {
                const isExpanded = expandedMobileSection === key;
                
                // Handle simple links (like Contact Us)
                if ('isSimple' in menu && menu.isSimple) {
                  return (
                    <div key={key} className="border-b border-gray-100 pb-4 last:border-b-0">
                      <a
                        href={menu.href}
                        onClick={() => setMobileMenuOpen(false)}
                        className="w-full flex items-center font-bold text-[#1A1A40] text-lg hover:text-[#ED1C24] transition-colors"
                      >
                        <div className="w-1 h-4 bg-[#ED1C24] rounded-full mr-2"></div>
                        {menu.title}
                      </a>
                    </div>
                  );
                }

                // Handle complex mega menus
                return (
                  <div key={key} className="border-b border-gray-100 pb-4 last:border-b-0">
                    <button
                      onClick={() => setExpandedMobileSection(isExpanded ? null : key)}
                      className="w-full flex items-center justify-between font-bold text-[#1A1A40] mb-3 text-lg hover:text-[#ED1C24] transition-colors"
                    >
                      <div className="flex items-center">
                        <div className="w-1 h-4 bg-[#ED1C24] rounded-full mr-2"></div>
                        {menu.title}
                      </div>
                      <ChevronDown className={`h-4 w-4 transition-transform duration-200 ${
                        isExpanded ? 'rotate-180' : ''
                      }`} />
                    </button>
                    
                    {isExpanded && 'sections' in menu && (
                      <div className="transition-all duration-200 ease-in-out">
                        {menu.sections.map((section: any, index: number) => (
                          <div key={index} className="ml-3 mb-4 last:mb-0">
                            <a 
                              href={section.href}
                              onClick={() => setMobileMenuOpen(false)}
                              className="font-semibold text-gray-700 mb-2 text-base flex items-center hover:text-[#ED1C24] transition-colors"
                            >
                              {section.icon && <section.icon className="h-4 w-4 mr-2 text-[#ED1C24]" />}
                              {section.title}
                            </a>
                            
                            {/* Show sub-links if they exist */}
                            {section.subLinks && (
                              <div className="ml-6 mt-2 space-y-2">
                                {section.subLinks.map((subLink: any, subIndex: number) => (
                                  <div key={subIndex}>
                                    <h5 className="font-medium text-gray-600 text-xs mb-1 uppercase tracking-wide">{subLink.name}</h5>
                                    <ul className="ml-2 space-y-1">
                                      {subLink.items.map((item: any, itemIndex: number) => (
                                        <li key={itemIndex}>
                                          <a 
                                            href={typeof item === 'string' ? "#" : item.href || "#"}
                                            onClick={() => setMobileMenuOpen(false)}
                                            className="text-gray-500 text-xs flex items-center py-1 cursor-pointer hover:text-[#ED1C24] transition-colors"
                                          >
                                            <div className="w-1 h-1 bg-gray-400 rounded-full mr-2"></div>
                                            {typeof item === 'string' ? item : item.name}
                                          </a>
                                        </li>
                                      ))}
                                    </ul>
                                  </div>
                                ))}
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                );
              })}
            </div>
          </div>
        )}
      </nav>
    </div>
  );
}