import { useState } from "react";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { z } from "zod";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Checkbox } from "@/components/ui/checkbox";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { Separator } from "@/components/ui/separator";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { useMutation } from "@tanstack/react-query";
import { apiRequest } from "@/lib/queryClient";
import { 
  Building, 
  User, 
  Phone, 
  Mail, 
  MapPin, 
  FileText, 
  Award,
  Camera,
  Upload,
  CheckCircle,
  AlertCircle
} from "lucide-react";

const accreditationApplicationSchema = z.object({
  applicationType: z.enum(["training_center", "testing_center", "both"]),
  organizationName: z.string().min(1, "Organization name is required"),
  contactPerson: z.string().min(1, "Contact person is required"),
  contactEmail: z.string().email("Valid email is required"),
  contactPhone: z.string().min(1, "Phone number is required"),
  address: z.string().min(1, "Address is required"),
  city: z.string().min(1, "City is required"),
  country: z.string().min(1, "Country is required"),
  businessLicense: z.string().min(1, "Business license number is required"),
  taxRegistration: z.string().min(1, "Tax registration number is required"),
  organizationType: z.enum(["private", "public", "non_profit"]),
  establishedYear: z.number().min(1900).max(new Date().getFullYear()),
  numberOfEmployees: z.number().min(1),
  facilitiesDescription: z.string().min(1, "Facilities description is required"),
  technicalInfrastructure: z.object({
    computers: z.number().min(1),
    internetCapacity: z.string().min(1),
    backupPower: z.boolean(),
    securitySystems: z.boolean(),
    accessibilityFeatures: z.boolean(),
  }),
  qualifiedPersonnel: z.array(z.object({
    name: z.string().min(1),
    position: z.string().min(1),
    qualifications: z.string().min(1),
    experience: z.string().min(1),
  })).min(1, "At least one qualified person is required"),
  certificationPrograms: z.array(z.string()).min(1, "Select at least one certification program"),
  operatingHours: z.object({
    weekdays: z.string().min(1),
    weekends: z.string().min(1),
    holidays: z.string().min(1),
  }),
  capacityInformation: z.object({
    simultaneousExams: z.number().min(1),
    dailyCapacity: z.number().min(1),
    monthlyCapacity: z.number().min(1),
  }),
  qualityAssurance: z.object({
    hasQualityPolicy: z.boolean(),
    qualityPolicyDescription: z.string().optional(),
    hasComplaintProcess: z.boolean(),
    complaintProcessDescription: z.string().optional(),
  }),
  supportingDocuments: z.array(z.string()).min(1, "Upload at least one supporting document"),
  facilityPhotos: z.array(z.string()).min(1, "Upload at least one facility photo"),
  agreementToTerms: z.boolean().refine((val) => val === true, {
    message: "You must agree to the terms and conditions",
  }),
  declarationOfAccuracy: z.boolean().refine((val) => val === true, {
    message: "You must declare the accuracy of information",
  }),
});

type AccreditationApplicationFormData = z.infer<typeof accreditationApplicationSchema>;

interface AccreditationApplicationFormProps {
  onSubmit: (data: AccreditationApplicationFormData) => void;
  isLoading?: boolean;
  onCancel?: () => void;
  initialApplicationType?: string;
}

export default function AccreditationApplicationForm({ 
  onSubmit, 
  isLoading = false,
  onCancel,
  initialApplicationType = "training_center" 
}: AccreditationApplicationFormProps) {
  const [currentStep, setCurrentStep] = useState(1);
  const [personnelList, setPersonnelList] = useState([{ name: "", position: "", qualifications: "", experience: "" }]);
  const { toast } = useToast();

  const form = useForm<AccreditationApplicationFormData>({
    resolver: zodResolver(accreditationApplicationSchema),
    defaultValues: {
      applicationType: initialApplicationType as any,
      organizationType: "private",
      establishedYear: new Date().getFullYear(),
      numberOfEmployees: 1,
      technicalInfrastructure: {
        computers: 1,
        internetCapacity: "",
        backupPower: false,
        securitySystems: false,
        accessibilityFeatures: false,
      },
      qualifiedPersonnel: [{ name: "", position: "", qualifications: "", experience: "" }],
      certificationPrograms: [],
      operatingHours: {
        weekdays: "",
        weekends: "",
        holidays: "",
      },
      capacityInformation: {
        simultaneousExams: 1,
        dailyCapacity: 1,
        monthlyCapacity: 1,
      },
      qualityAssurance: {
        hasQualityPolicy: false,
        qualityPolicyDescription: "",
        hasComplaintProcess: false,
        complaintProcessDescription: "",
      },
      supportingDocuments: [],
      facilityPhotos: [],
      agreementToTerms: false,
      declarationOfAccuracy: false,
    },
  });

  const watchedApplicationType = form.watch("applicationType");
  const watchedQualityPolicy = form.watch("qualityAssurance.hasQualityPolicy");
  const watchedComplaintProcess = form.watch("qualityAssurance.hasComplaintProcess");

  const addPersonnel = () => {
    setPersonnelList([...personnelList, { name: "", position: "", qualifications: "", experience: "" }]);
  };

  const removePersonnel = (index: number) => {
    if (personnelList.length > 1) {
      const newList = personnelList.filter((_, i) => i !== index);
      setPersonnelList(newList);
      form.setValue("qualifiedPersonnel", newList);
    }
  };

  const updatePersonnel = (index: number, field: string, value: string) => {
    const newList = [...personnelList];
    newList[index] = { ...newList[index], [field]: value };
    setPersonnelList(newList);
    form.setValue("qualifiedPersonnel", newList);
  };

  const nextStep = () => {
    if (currentStep < 5) {
      setCurrentStep(currentStep + 1);
    }
  };

  const prevStep = () => {
    if (currentStep > 1) {
      setCurrentStep(currentStep - 1);
    }
  };

  const handleSubmit = (data: AccreditationApplicationFormData) => {
    onSubmit(data);
  };

  const certificationPrograms = [
    "Strategic Leadership",
    "Project Management",
    "Digital Marketing",
    "Data Analytics",
    "Financial Management",
    "Human Resources",
    "Quality Management",
    "Supply Chain Management",
    "Risk Management",
    "Business Development",
  ];

  const countries = [
    "United States", "Canada", "United Kingdom", "Germany", "France", "Australia",
    "United Arab Emirates", "Saudi Arabia", "Egypt", "Jordan", "Lebanon", "Kuwait",
    "Qatar", "Bahrain", "Oman", "Morocco", "Tunisia", "Algeria", "Libya", "Sudan",
    "Iraq", "Yemen", "Palestine", "Syria", "Somalia", "Djibouti", "Mauritania",
    "India", "China", "Japan", "Singapore", "Malaysia", "Indonesia", "Thailand",
    "Philippines", "Vietnam", "South Korea", "Brazil", "Mexico", "Argentina",
    "Chile", "Colombia", "Peru", "South Africa", "Nigeria", "Ghana", "Kenya",
    "Ethiopia", "Tanzania", "Uganda", "Rwanda", "Botswana", "Zambia", "Zimbabwe"
  ];

  const renderStep = () => {
    switch (currentStep) {
      case 1:
        return (
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-2xl font-bold">Basic Information</h2>
              <p className="text-muted-foreground">Tell us about your organization</p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <FormField
                control={form.control}
                name="applicationType"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Application Type</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue placeholder="Select application type" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="training_center">Training Center</SelectItem>
                        <SelectItem value="testing_center">Testing Center</SelectItem>
                        <SelectItem value="both">Both Training & Testing</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="organizationType"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Organization Type</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue placeholder="Select organization type" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="private">Private</SelectItem>
                        <SelectItem value="public">Public</SelectItem>
                        <SelectItem value="non_profit">Non-Profit</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="organizationName"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Organization Name</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter organization name" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="establishedYear"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Established Year</FormLabel>
                    <FormControl>
                      <Input 
                        type="number" 
                        placeholder="YYYY" 
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value))}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="numberOfEmployees"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Number of Employees</FormLabel>
                    <FormControl>
                      <Input 
                        type="number" 
                        placeholder="Number of employees" 
                        {...field}
                        onChange={(e) => field.onChange(parseInt(e.target.value))}
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="businessLicense"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Business License Number</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter business license number" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="taxRegistration"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Tax Registration Number</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter tax registration number" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>
          </div>
        );

      case 2:
        return (
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-2xl font-bold">Contact Information</h2>
              <p className="text-muted-foreground">How can we reach you?</p>
            </div>

            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
              <FormField
                control={form.control}
                name="contactPerson"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Contact Person</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter contact person name" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="contactEmail"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Contact Email</FormLabel>
                    <FormControl>
                      <Input type="email" placeholder="Enter email address" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="contactPhone"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Contact Phone</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter phone number" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="country"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Country</FormLabel>
                    <Select onValueChange={field.onChange} defaultValue={field.value}>
                      <FormControl>
                        <SelectTrigger>
                          <SelectValue placeholder="Select country" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {countries.map((country) => (
                          <SelectItem key={country} value={country}>
                            {country}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="city"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>City</FormLabel>
                    <FormControl>
                      <Input placeholder="Enter city" {...field} />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />
            </div>

            <FormField
              control={form.control}
              name="address"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Full Address</FormLabel>
                  <FormControl>
                    <Textarea placeholder="Enter complete address" {...field} />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />
          </div>
        );

      case 3:
        return (
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-2xl font-bold">Facilities & Infrastructure</h2>
              <p className="text-muted-foreground">Tell us about your facilities and technical capabilities</p>
            </div>

            <FormField
              control={form.control}
              name="facilitiesDescription"
              render={({ field }) => (
                <FormItem>
                  <FormLabel>Facilities Description</FormLabel>
                  <FormControl>
                    <Textarea 
                      placeholder="Describe your facilities including size, layout, and key features" 
                      {...field} 
                    />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )}
            />

            <Card>
              <CardHeader>
                <CardTitle>Technical Infrastructure</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <FormField
                    control={form.control}
                    name="technicalInfrastructure.computers"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Number of Computers</FormLabel>
                        <FormControl>
                          <Input 
                            type="number" 
                            placeholder="Number of computers" 
                            {...field}
                            onChange={(e) => field.onChange(parseInt(e.target.value))}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="technicalInfrastructure.internetCapacity"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Internet Capacity</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., 100 Mbps" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <FormField
                    control={form.control}
                    name="technicalInfrastructure.backupPower"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                        <FormControl>
                          <Checkbox
                            checked={field.value}
                            onCheckedChange={field.onChange}
                          />
                        </FormControl>
                        <div className="space-y-1 leading-none">
                          <FormLabel>Backup Power</FormLabel>
                          <p className="text-sm text-muted-foreground">
                            UPS or generator available
                          </p>
                        </div>
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="technicalInfrastructure.securitySystems"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                        <FormControl>
                          <Checkbox
                            checked={field.value}
                            onCheckedChange={field.onChange}
                          />
                        </FormControl>
                        <div className="space-y-1 leading-none">
                          <FormLabel>Security Systems</FormLabel>
                          <p className="text-sm text-muted-foreground">
                            CCTV, access control, etc.
                          </p>
                        </div>
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="technicalInfrastructure.accessibilityFeatures"
                    render={({ field }) => (
                      <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                        <FormControl>
                          <Checkbox
                            checked={field.value}
                            onCheckedChange={field.onChange}
                          />
                        </FormControl>
                        <div className="space-y-1 leading-none">
                          <FormLabel>Accessibility Features</FormLabel>
                          <p className="text-sm text-muted-foreground">
                            Disabled access, ramps, etc.
                          </p>
                        </div>
                      </FormItem>
                    )}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Capacity Information</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <FormField
                    control={form.control}
                    name="capacityInformation.simultaneousExams"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Simultaneous Exams</FormLabel>
                        <FormControl>
                          <Input 
                            type="number" 
                            placeholder="Number" 
                            {...field}
                            onChange={(e) => field.onChange(parseInt(e.target.value))}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="capacityInformation.dailyCapacity"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Daily Capacity</FormLabel>
                        <FormControl>
                          <Input 
                            type="number" 
                            placeholder="Number" 
                            {...field}
                            onChange={(e) => field.onChange(parseInt(e.target.value))}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="capacityInformation.monthlyCapacity"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Monthly Capacity</FormLabel>
                        <FormControl>
                          <Input 
                            type="number" 
                            placeholder="Number" 
                            {...field}
                            onChange={(e) => field.onChange(parseInt(e.target.value))}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Operating Hours</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <FormField
                    control={form.control}
                    name="operatingHours.weekdays"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Weekdays</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., 9:00 AM - 5:00 PM" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="operatingHours.weekends"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Weekends</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., 10:00 AM - 4:00 PM" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />

                  <FormField
                    control={form.control}
                    name="operatingHours.holidays"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Holidays</FormLabel>
                        <FormControl>
                          <Input placeholder="e.g., Closed or Limited Hours" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                </div>
              </CardContent>
            </Card>
          </div>
        );

      case 4:
        return (
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-2xl font-bold">Personnel & Programs</h2>
              <p className="text-muted-foreground">Tell us about your team and certification programs</p>
            </div>

            <Card>
              <CardHeader>
                <div className="flex justify-between items-center">
                  <CardTitle>Qualified Personnel</CardTitle>
                  <Button type="button" variant="outline" onClick={addPersonnel}>
                    <Plus className="w-4 h-4 mr-2" />
                    Add Personnel
                  </Button>
                </div>
              </CardHeader>
              <CardContent className="space-y-4">
                {personnelList.map((person, index) => (
                  <div key={index} className="grid grid-cols-1 md:grid-cols-2 gap-4 p-4 border rounded-lg">
                    <div className="md:col-span-2 flex justify-between items-center">
                      <h4 className="font-semibold">Person {index + 1}</h4>
                      {personnelList.length > 1 && (
                        <Button 
                          type="button" 
                          variant="ghost" 
                          size="sm"
                          onClick={() => removePersonnel(index)}
                        >
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      )}
                    </div>
                    <div>
                      <Label>Name</Label>
                      <Input
                        placeholder="Full name"
                        value={person.name}
                        onChange={(e) => updatePersonnel(index, "name", e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>Position</Label>
                      <Input
                        placeholder="Job title/position"
                        value={person.position}
                        onChange={(e) => updatePersonnel(index, "position", e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>Qualifications</Label>
                      <Input
                        placeholder="Degrees, certifications"
                        value={person.qualifications}
                        onChange={(e) => updatePersonnel(index, "qualifications", e.target.value)}
                      />
                    </div>
                    <div>
                      <Label>Experience</Label>
                      <Input
                        placeholder="Years and type of experience"
                        value={person.experience}
                        onChange={(e) => updatePersonnel(index, "experience", e.target.value)}
                      />
                    </div>
                  </div>
                ))}
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Certification Programs</CardTitle>
                <CardDescription>
                  Select the certification programs you want to offer
                </CardDescription>
              </CardHeader>
              <CardContent>
                <FormField
                  control={form.control}
                  name="certificationPrograms"
                  render={() => (
                    <FormItem>
                      <div className="grid grid-cols-2 md:grid-cols-3 gap-3">
                        {certificationPrograms.map((program) => (
                          <FormField
                            key={program}
                            control={form.control}
                            name="certificationPrograms"
                            render={({ field }) => {
                              return (
                                <FormItem
                                  key={program}
                                  className="flex flex-row items-start space-x-3 space-y-0"
                                >
                                  <FormControl>
                                    <Checkbox
                                      checked={field.value?.includes(program)}
                                      onCheckedChange={(checked) => {
                                        return checked
                                          ? field.onChange([...field.value, program])
                                          : field.onChange(
                                              field.value?.filter(
                                                (value) => value !== program
                                              )
                                            );
                                      }}
                                    />
                                  </FormControl>
                                  <FormLabel className="text-sm font-normal">
                                    {program}
                                  </FormLabel>
                                </FormItem>
                              );
                            }}
                          />
                        ))}
                      </div>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Quality Assurance</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <FormField
                  control={form.control}
                  name="qualityAssurance.hasQualityPolicy"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                      <FormControl>
                        <Checkbox
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Do you have a quality policy?</FormLabel>
                        <p className="text-sm text-muted-foreground">
                          Quality management system in place
                        </p>
                      </div>
                    </FormItem>
                  )}
                />

                {watchedQualityPolicy && (
                  <FormField
                    control={form.control}
                    name="qualityAssurance.qualityPolicyDescription"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Quality Policy Description</FormLabel>
                        <FormControl>
                          <Textarea placeholder="Describe your quality policy" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                )}

                <FormField
                  control={form.control}
                  name="qualityAssurance.hasComplaintProcess"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                      <FormControl>
                        <Checkbox
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Do you have a complaint handling process?</FormLabel>
                        <p className="text-sm text-muted-foreground">
                          System for handling client complaints
                        </p>
                      </div>
                    </FormItem>
                  )}
                />

                {watchedComplaintProcess && (
                  <FormField
                    control={form.control}
                    name="qualityAssurance.complaintProcessDescription"
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel>Complaint Process Description</FormLabel>
                        <FormControl>
                          <Textarea placeholder="Describe your complaint handling process" {...field} />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                )}
              </CardContent>
            </Card>
          </div>
        );

      case 5:
        return (
          <div className="space-y-6">
            <div className="text-center">
              <h2 className="text-2xl font-bold">Documents & Declaration</h2>
              <p className="text-muted-foreground">Upload required documents and confirm your application</p>
            </div>

            <Card>
              <CardHeader>
                <CardTitle>Required Documents</CardTitle>
                <CardDescription>
                  Please upload the following documents (PDF, DOC, or images accepted)
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Business License</p>
                    <p className="text-xs text-gray-500">Click to upload or drag and drop</p>
                  </div>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Tax Registration</p>
                    <p className="text-xs text-gray-500">Click to upload or drag and drop</p>
                  </div>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Insurance Certificate</p>
                    <p className="text-xs text-gray-500">Click to upload or drag and drop</p>
                  </div>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Upload className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Staff Qualifications</p>
                    <p className="text-xs text-gray-500">Click to upload or drag and drop</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Facility Photos</CardTitle>
                <CardDescription>
                  Upload photos of your facility (minimum 3 photos required)
                </CardDescription>
              </CardHeader>
              <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Camera className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Main Entrance</p>
                    <p className="text-xs text-gray-500">Click to upload photo</p>
                  </div>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Camera className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Exam Room</p>
                    <p className="text-xs text-gray-500">Click to upload photo</p>
                  </div>
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 text-center">
                    <Camera className="w-8 h-8 mx-auto mb-2 text-gray-400" />
                    <p className="text-sm text-gray-600">Reception Area</p>
                    <p className="text-xs text-gray-500">Click to upload photo</p>
                  </div>
                </div>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Declaration & Agreement</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <FormField
                  control={form.control}
                  name="agreementToTerms"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                      <FormControl>
                        <Checkbox
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Agreement to Terms and Conditions</FormLabel>
                        <p className="text-sm text-muted-foreground">
                          I agree to comply with all IBDL accreditation standards, requirements, and ongoing obligations.
                        </p>
                      </div>
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="declarationOfAccuracy"
                  render={({ field }) => (
                    <FormItem className="flex flex-row items-start space-x-3 space-y-0 rounded-md border p-4">
                      <FormControl>
                        <Checkbox
                          checked={field.value}
                          onCheckedChange={field.onChange}
                        />
                      </FormControl>
                      <div className="space-y-1 leading-none">
                        <FormLabel>Declaration of Accuracy</FormLabel>
                        <p className="text-sm text-muted-foreground">
                          I declare that all information provided in this application is true, accurate, and complete.
                        </p>
                      </div>
                    </FormItem>
                  )}
                />
              </CardContent>
            </Card>
          </div>
        );

      default:
        return null;
    }
  };

  const getStepIcon = (step: number) => {
    if (step < currentStep) {
      return <CheckCircle className="w-6 h-6 text-green-600" />;
    } else if (step === currentStep) {
      return <div className="w-6 h-6 bg-primary rounded-full flex items-center justify-center text-white text-sm font-bold">{step}</div>;
    } else {
      return <div className="w-6 h-6 bg-gray-200 rounded-full flex items-center justify-center text-gray-600 text-sm font-bold">{step}</div>;
    }
  };

  const stepTitles = [
    "Basic Information",
    "Contact Details",
    "Facilities",
    "Personnel & Programs",
    "Documents & Declaration"
  ];

  return (
    <div className="max-w-4xl mx-auto p-6">
      <div className="mb-8">
        <h1 className="text-3xl font-bold text-center mb-2">
          {watchedApplicationType === "training_center" ? "Training Center" : 
           watchedApplicationType === "testing_center" ? "Testing Center" : 
           "Training & Testing Center"} Accreditation Application
        </h1>
        <p className="text-center text-muted-foreground">
          Apply for IBDL accreditation to become an authorized {watchedApplicationType.replace('_', ' ')}
        </p>
      </div>

      {/* Progress Steps */}
      <div className="mb-8">
        <div className="flex justify-between items-center">
          {stepTitles.map((title, index) => (
            <div key={index} className="flex flex-col items-center">
              {getStepIcon(index + 1)}
              <div className={`text-xs mt-2 text-center ${index + 1 === currentStep ? 'text-primary font-semibold' : 'text-gray-500'}`}>
                {title}
              </div>
            </div>
          ))}
        </div>
        <div className="mt-4 bg-gray-200 rounded-full h-2">
          <div 
            className="bg-primary h-2 rounded-full transition-all duration-300"
            style={{ width: `${(currentStep / 5) * 100}%` }}
          />
        </div>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-6">
          {renderStep()}

          <div className="flex justify-between pt-6">
            <Button
              type="button"
              variant="outline"
              onClick={currentStep === 1 ? onCancel : prevStep}
              disabled={isLoading}
            >
              {currentStep === 1 ? "Cancel" : "Previous"}
            </Button>
            
            {currentStep < 5 ? (
              <Button type="button" onClick={nextStep}>
                Next
              </Button>
            ) : (
              <Button type="submit" disabled={isLoading}>
                {isLoading ? "Submitting..." : "Submit Application"}
              </Button>
            )}
          </div>
        </form>
      </Form>
    </div>
  );
}