import { Switch, Route, useLocation } from "wouter";
import { queryClient } from "./lib/queryClient";
import { QueryClientProvider } from "@tanstack/react-query";
import { useState, useEffect } from "react";
import { Toaster } from "@/components/ui/toaster";
import { TooltipProvider } from "@/components/ui/tooltip";
import { useAuth } from "@/hooks/useAuth";
import Landing from "@/pages/Landing";
import Login from "@/pages/Login";
import ZeroAdminLogin from "@/pages/ZeroAdminLogin";
import AdminLogin from "@/pages/AdminLogin";
import GetStarted from "@/pages/GetStarted";
import LearnerDashboard from "@/pages/dashboard/LearnerDashboard";
import CorporateDashboard from "@/pages/dashboard/CorporateDashboard";
import PartnerDashboard from "@/pages/dashboard/PartnerDashboard";
import AdminDashboard from "@/pages/dashboard/AdminDashboard";
import CourseCatalog from "@/pages/courses/CourseCatalog";
import CourseDetail from "@/pages/courses/CourseDetail";
import InteractiveLearning from "@/pages/courses/InteractiveLearning";
import AssessmentList from "@/pages/assessments/AssessmentList";
import AssessmentDetail from "@/pages/assessments/AssessmentDetail";
import AssessmentTaking from "@/pages/assessments/AssessmentTaking";
import ExaminationCenter from "@/pages/assessments/ExaminationCenter";
import ExaminationTaking from "@/pages/assessments/ExaminationTaking";
import ExaminationResults from "@/pages/assessments/ExaminationResults";
import CertificateList from "@/pages/certificates/CertificateList";
import AccreditationManagement from "@/pages/AccreditationManagement";
import AccreditationApplication from "@/pages/AccreditationApplication";
import TrainingCenterApplication from "@/pages/TrainingCenterApplication";
import ApplicantDashboard from "@/pages/ApplicantDashboard";
import CorporateAccountManagement from "@/pages/CorporateAccountManagement";
import QuestionBankManagement from "@/pages/QuestionBankManagement";
import SystemSetup from "@/pages/SystemSetup";
import Examinations from "@/pages/Examinations";
import Assessments from "@/pages/Assessments";
import ELearning from "@/pages/ELearning";
import EBooks from "@/pages/EBooks";
import QAAccreditation from "@/pages/QAAccreditation";
import DashboardLayout from "@/components/layouts/DashboardLayout";
import NotFound from "@/pages/not-found";
import Loading from "@/components/ui/loading";
import About from "@/pages/About";
import Store from "@/pages/Store";
import News from "@/pages/News";
import Resources from "@/pages/Resources";
import Magazine from "@/pages/Magazine";
import Credentials from "@/pages/Credentials";
import Solutions from "@/pages/Solutions";
import CMS from "@/pages/CMS";
import Strategy from "@/pages/Strategy";
import Qualifications from "@/pages/Qualifications";
import Contact from "@/pages/Contact";
import CMSStrategy from "@/pages/CMSStrategy";
import CMSQualifications from "@/pages/CMSQualifications";
import CMSContact from "@/pages/CMSContact";
import CMSAbout from "./pages/CMSAbout";
import CMSSolutions from "./pages/CMSSolutions";
import CMSAccreditation from "./pages/CMSAccreditation";
import WebsiteAdmin from "@/pages/WebsiteAdmin";

// Who We Are pages
import OurStory from "@/pages/WhoWeAre/OurStory";
import LeadershipGovernance from "@/pages/WhoWeAre/LeadershipGovernance";
import OurImpact from "@/pages/WhoWeAre/OurImpact";
import KnowledgeFramework from "@/pages/WhoWeAre/KnowledgeFramework";
import PartnersInExcellence from "@/pages/WhoWeAre/PartnersInExcellence";

// Our Products - Core Qualifications
import IBDLEssential from "@/pages/OurProducts/Qualifications/CoreQualifications/IBDLEssential";
import IBDLProfessional from "@/pages/OurProducts/Qualifications/CoreQualifications/IBDLProfessional";
import IBDLMaster from "@/pages/OurProducts/Qualifications/CoreQualifications/IBDLMaster";

// Our Products - Specialized Qualifications
import CertifiedEducator from "@/pages/OurProducts/Qualifications/SpecializedQualifications/CertifiedEducator";
import CertifiedIndustrialEngineer from "@/pages/OurProducts/Qualifications/SpecializedQualifications/CertifiedIndustrialEngineer";
import CNPOPractitioner from "@/pages/OurProducts/Qualifications/SpecializedQualifications/CNPOPractitioner";
import CNPOManagementProfessional from "@/pages/OurProducts/Qualifications/SpecializedQualifications/CNPOManagementProfessional";

// Our Products - Foresight Qualifications
import ForesightAssociate from "@/pages/OurProducts/Qualifications/ForesightQualifications/ForesightAssociate";
import ForesightPractitioner from "@/pages/OurProducts/Qualifications/ForesightQualifications/ForesightPractitioner";
import ForesightStrategist from "@/pages/OurProducts/Qualifications/ForesightQualifications/ForesightStrategist";

// Products pages
import ProductQualifications from "@/pages/Products/Qualifications";
import LearningGames from "@/pages/Products/LearningGames";
import AssessmentServices from "@/pages/Products/AssessmentServices";
import ExecutiveEducation from "@/pages/Products/ExecutiveEducation";
import VentureHub from "@/pages/Products/VentureHub";

// Learning Games subtab pages
import WinVsWar from "@/pages/Products/LearningGames/WinVsWar";
import MasterBoardGame from "@/pages/Products/LearningGames/MasterBoardGame";
import Sparta from "@/pages/Products/LearningGames/Sparta";
import MogulCEO from "@/pages/Products/LearningGames/MogulCEO";
import MavenMarketing from "@/pages/Products/LearningGames/MavenMarketing";
import Micromatic from "@/pages/Products/LearningGames/Micromatic";
import PitchDeck from "@/pages/Products/LearningGames/PitchDeck";
import SynergyStack from "@/pages/Products/LearningGames/SynergyStack";

// Venture Hub subtab pages
import AngelPlaybookBootcamp from "@/pages/Products/VentureHub/AngelPlaybookBootcamp";
import CapitalRaiseBootcamp from "@/pages/Products/VentureHub/CapitalRaiseBootcamp";
import DataRoomPreparationBootcamp from "@/pages/Products/VentureHub/DataRoomPreparationBootcamp";

// Venture Lab subtab pages
import CertifiedDisciplinedEntrepreneur from "@/pages/Products/VentureLab/CertifiedDisciplinedEntrepreneur";
import CertifiedIncubationManagement from "@/pages/Products/VentureLab/CertifiedIncubationManagement";

// Assessment Services subtab pages
import PQP from "@/pages/Products/AssessmentServices/PQP";
import CPAT from "@/pages/Products/AssessmentServices/CPAT";

// Services pages
import AccreditationServices from "@/pages/Services/AccreditationServices";
import BePartOfOurStory from "@/pages/Services/BePartOfOurStory";

// Services subtab pages
import TrainingProvider from "@/pages/Services/AccreditationServices/TrainingProvider";
import ExamProvider from "@/pages/Services/AccreditationServices/ExamProvider";
import CorporateGovernment from "@/pages/Services/AccreditationServices/CorporateGovernment";
import ProfessionalTrainer from "@/pages/Services/AccreditationServices/ProfessionalTrainer";
import ProfessionalCourse from "@/pages/Services/AccreditationServices/ProfessionalCourse";
import PartnerWithUs from "@/pages/Services/BePartOfOurStory/PartnerWithUs";
import JoinGlobalCommunity from "@/pages/Services/BePartOfOurStory/JoinGlobalCommunity";

// Resources pages
import Newsroom from "@/pages/Resources/Newsroom";
import LearningEdit from "@/pages/Resources/LearningEdit";
import ProgramBrochures from "@/pages/Resources/ProgramBrochures";
import EventsConferences from "@/pages/Resources/EventsConferences";

function Router() {
  const { isAuthenticated, isLoading, user } = useAuth();
  const [location, setLocation] = useLocation();
  const [currentUser, setCurrentUser] = useState<any>(null);
  const [showLogin, setShowLogin] = useState(false);
  const [showGetStarted, setShowGetStarted] = useState(false);

  // Handle login success and redirect to appropriate dashboard
  const handleLoginSuccess = (userData: any, dashboardRoute: string) => {
    setCurrentUser(userData);
    setShowLogin(false); // Hide login page
    queryClient.invalidateQueries({ queryKey: ["backend/api/auth/user"] });
    setLocation("/"); // Navigate to home, which will redirect to appropriate dashboard
  };

  // Show login when Sign In button is clicked
  const handleShowLogin = () => {
    setShowLogin(true);
    setShowGetStarted(false);
  };

  // Show get started when Get Started button is clicked
  const handleShowGetStarted = () => {
    setShowGetStarted(true);
    setShowLogin(false);
  };

  // Handle role selection from GetStarted page
  const handleRoleSelect = (role: string) => {
    // Store selected role and redirect to login
    // In a real app, this would go to a registration form first
    setShowGetStarted(false);
    setShowLogin(true);
  };

  // Handle back to home from login/get started pages
  const handleBackToHome = () => {
    setShowLogin(false);
    setShowGetStarted(false);
  };

  if (isLoading) {
    return (
      <div className="min-h-screen flex items-center justify-center bg-white">
        <Loading size="lg" text="Loading IBDL Platform..." />
      </div>
    );
  }

  // Show GetStarted page when requested
  if (showGetStarted) {
    return <GetStarted onRoleSelect={handleRoleSelect} onBack={handleBackToHome} />;
  }

  // Show login page when requested or when trying to access protected routes without auth
  // Allow public access to accreditation application and external pages
  const publicRoutes = [
    "/", "/get-started", "/accreditation-application", "/training-center-application", 
    "/about", "/store", "/news", "/resources", "/magazine", "/credentials", "/solutions", 
    "/strategy", "/qualifications", "/contact", "/accreditation", "/zeroadmin", "/admin",
    // Who We Are routes
    "/who-we-are/our-story", "/who-we-are/leadership-governance", "/who-we-are/our-impact", 
    "/who-we-are/knowledge-framework", "/who-we-are/partners-in-excellence",
    // Our Products - Qualifications routes
    "/our-products/qualifications/core-qualifications/ibdl-essential",
    "/our-products/qualifications/core-qualifications/ibdl-professional",
    "/our-products/qualifications/core-qualifications/ibdl-master",
    "/our-products/qualifications/specialized-qualifications/certified-educator",
    "/our-products/qualifications/specialized-qualifications/certified-industrial-engineer",
    "/our-products/qualifications/specialized-qualifications/cnpo-practitioner",
    "/our-products/qualifications/specialized-qualifications/cnpo-management-professional",
    "/our-products/qualifications/foresight-qualifications/foresight-associate",
    "/our-products/qualifications/foresight-qualifications/foresight-practitioner",
    "/our-products/qualifications/foresight-qualifications/foresight-strategist",
    // Products routes
    "/products/qualifications", "/products/learning-games", "/products/assessment-services",
    "/products/executive-education", "/products/venture-hub",
    // Learning Games subtab routes
    "/products/learning-games/win-vs-war", "/products/learning-games/master-board-game",
    "/products/learning-games/sparta", "/products/learning-games/mogul-ceo", 
    "/products/learning-games/maven-marketing", "/products/learning-games/micromatic",
    "/products/learning-games/pitch-deck", "/products/learning-games/synergy-stack",
    // Venture Hub subtab routes
    "/products/venture-hub/angel-playbook-bootcamp", "/products/venture-hub/capital-raise-bootcamp",
    "/products/venture-hub/data-room-preparation-bootcamp",
    // Venture Lab subtab routes
    "/products/venture-lab/certified-disciplined-entrepreneur", 
    "/products/venture-lab/certified-incubation-management",
    // Assessment Services subtab routes
    "/products/assessment-services/pqp", "/products/assessment-services/cpat",
    // Services routes
    "/services/accreditation-services", "/services/be-part-of-our-story",
    // Services subtab routes
    "/services/accreditation-services/training-provider", "/services/accreditation-services/exam-provider",
    "/services/accreditation-services/corporate-government", 
    "/services/accreditation-services/professional-trainer", "/services/accreditation-services/professional-course",
    "/services/be-part-of-our-story/partner-with-us", "/services/be-part-of-our-story/join-global-community",
    // Resources routes
    "/resources/newsroom", "/resources/learning-edit", "/resources/program-brochures", 
    "/resources/events-conferences"
  ];
  if (showLogin || (!publicRoutes.includes(location) && !isAuthenticated)) {
    return <Login />;
  }

  return (
    <Switch>
      {/* Admin login routes - completely isolated */}
      <Route path="/zeroadmin">
        {() => <ZeroAdminLogin />}
      </Route>
      <Route path="/admin">
        {() => <AdminLogin />}
      </Route>
      <Route path="/login">
        {() => <Login />}
      </Route>
      <Route path="/get-started">
        {() => <GetStarted onRoleSelect={handleRoleSelect} onBack={handleBackToHome} />}
      </Route>
      
      {/* Public routes - Always available */}
      <Route path="/accreditation-application" component={AccreditationApplication} />
      <Route path="/training-center-application" component={TrainingCenterApplication} />
      <Route path="/about" component={About} />
      <Route path="/store" component={Store} />
      <Route path="/news" component={News} />
      <Route path="/resources" component={Resources} />
      <Route path="/magazine" component={Magazine} />
      <Route path="/credentials" component={Credentials} />
      <Route path="/solutions" component={Solutions} />
      <Route path="/strategy" component={CMSStrategy} />
      <Route path="/qualifications" component={CMSQualifications} />
      <Route path="/contact" component={CMSContact} />
      <Route path="/accreditation" component={CMSAccreditation} />
      
      {/* Who We Are Routes */}
      <Route path="/who-we-are/our-story" component={OurStory} />
      <Route path="/who-we-are/leadership-governance" component={LeadershipGovernance} />
      <Route path="/who-we-are/our-impact" component={OurImpact} />
      <Route path="/who-we-are/knowledge-framework" component={KnowledgeFramework} />
      <Route path="/who-we-are/partners-in-excellence" component={PartnersInExcellence} />
      
      {/* Our Products - Qualifications Routes */}
      {/* Core Qualifications */}
      <Route path="/our-products/qualifications/core-qualifications/ibdl-essential" component={IBDLEssential} />
      <Route path="/our-products/qualifications/core-qualifications/ibdl-professional" component={IBDLProfessional} />
      <Route path="/our-products/qualifications/core-qualifications/ibdl-master" component={IBDLMaster} />
      
      {/* Specialized Qualifications */}
      <Route path="/our-products/qualifications/specialized-qualifications/certified-educator" component={CertifiedEducator} />
      <Route path="/our-products/qualifications/specialized-qualifications/certified-industrial-engineer" component={CertifiedIndustrialEngineer} />
      <Route path="/our-products/qualifications/specialized-qualifications/cnpo-practitioner" component={CNPOPractitioner} />
      <Route path="/our-products/qualifications/specialized-qualifications/cnpo-management-professional" component={CNPOManagementProfessional} />
      
      {/* Foresight Qualifications */}
      <Route path="/our-products/qualifications/foresight-qualifications/foresight-associate" component={ForesightAssociate} />
      <Route path="/our-products/qualifications/foresight-qualifications/foresight-practitioner" component={ForesightPractitioner} />
      <Route path="/our-products/qualifications/foresight-qualifications/foresight-strategist" component={ForesightStrategist} />
      
      {/* Products Routes */}
      <Route path="/products/qualifications" component={ProductQualifications} />
      <Route path="/products/learning-games" component={LearningGames} />
      <Route path="/products/assessment-services" component={AssessmentServices} />
      <Route path="/products/executive-education" component={ExecutiveEducation} />
      <Route path="/products/venture-hub" component={VentureHub} />
      
      {/* Learning Games Subtab Routes */}
      <Route path="/products/learning-games/win-vs-war" component={WinVsWar} />
      <Route path="/products/learning-games/master-board-game" component={MasterBoardGame} />
      <Route path="/products/learning-games/sparta" component={Sparta} />
      <Route path="/products/learning-games/mogul-ceo" component={MogulCEO} />
      <Route path="/products/learning-games/maven-marketing" component={MavenMarketing} />
      <Route path="/products/learning-games/micromatic" component={Micromatic} />
      <Route path="/products/learning-games/pitch-deck" component={PitchDeck} />
      <Route path="/products/learning-games/synergy-stack" component={SynergyStack} />
      
      {/* Venture Hub Subtab Routes */}
      <Route path="/products/venture-hub/angel-playbook-bootcamp" component={AngelPlaybookBootcamp} />
      <Route path="/products/venture-hub/capital-raise-bootcamp" component={CapitalRaiseBootcamp} />
      <Route path="/products/venture-hub/data-room-preparation-bootcamp" component={DataRoomPreparationBootcamp} />
      
      {/* Venture Lab Subtab Routes */}
      <Route path="/products/venture-lab/certified-disciplined-entrepreneur" component={CertifiedDisciplinedEntrepreneur} />
      <Route path="/products/venture-lab/certified-incubation-management" component={CertifiedIncubationManagement} />
      
      {/* Assessment Services Subtab Routes */}
      <Route path="/products/assessment-services/pqp" component={PQP} />
      <Route path="/products/assessment-services/cpat" component={CPAT} />
      
      {/* Services Routes */}
      <Route path="/services/accreditation-services" component={AccreditationServices} />
      <Route path="/services/be-part-of-our-story" component={BePartOfOurStory} />
      
      {/* Services Subtab Routes */}
      <Route path="/services/accreditation-services/training-provider" component={TrainingProvider} />
      <Route path="/services/accreditation-services/exam-provider" component={ExamProvider} />
      <Route path="/services/accreditation-services/corporate-government" component={CorporateGovernment} />
      <Route path="/services/accreditation-services/professional-trainer" component={ProfessionalTrainer} />
      <Route path="/services/accreditation-services/professional-course" component={ProfessionalCourse} />
      <Route path="/services/be-part-of-our-story/partner-with-us" component={PartnerWithUs} />
      <Route path="/services/be-part-of-our-story/join-global-community" component={JoinGlobalCommunity} />
      
      {/* Resources Routes */}
      <Route path="/resources/newsroom" component={Newsroom} />
      <Route path="/resources/learning-edit" component={LearningEdit} />
      <Route path="/resources/program-brochures" component={ProgramBrochures} />
      <Route path="/resources/events-conferences" component={EventsConferences} />
      
      {!isAuthenticated ? (
        <Route path="/" component={() => <Landing onShowLogin={handleShowLogin} onShowGetStarted={handleShowGetStarted} />} />
      ) : (
        <>
          <Route path="/" component={() => {
            // Redirect to appropriate dashboard based on user type
            const userToCheck = user || currentUser;
            if (!userToCheck) return <Landing onShowLogin={handleShowLogin} onShowGetStarted={handleShowGetStarted} />;
            
            switch (userToCheck.userType) {
              case 'corporate_admin':
                return <CorporateDashboard />;
              case 'partner_admin':
                return <PartnerDashboard />;
              case 'system_admin':
                return <AdminDashboard />;
              case 'qa_admin':
                window.location.href = '/qa';
                return null;
              case 'website_admin':
                window.location.href = '/cms';
                return null;
              case 'applicant':
              case 'learner':
                return <ApplicantDashboard />;
              default:
                return <LearnerDashboard />;
            }
          }} />
          <Route path="/dashboard/learner" component={LearnerDashboard} />
          <Route path="/dashboard/corporate" component={CorporateDashboard} />
          <Route path="/dashboard/partner" component={PartnerDashboard} />
          <Route path="/dashboard/admin" component={AdminDashboard} />

          <Route path="/courses" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <CourseCatalog />
              </DashboardLayout>
            );
          }} />
          <Route path="/courses/:courseId" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <CourseDetail />
              </DashboardLayout>
            );
          }} />
          <Route path="/courses/:courseId/learn" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <InteractiveLearning />
              </DashboardLayout>
            );
          }} />
          <Route path="/assessments" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            
            // System admin gets the new Assessments management page
            if (userToCheck?.userType === 'system_admin') {
              return <Assessments />;
            }
            
            // Other users get the assessment list page
            return (
              <DashboardLayout userType={userType}>
                <AssessmentList />
              </DashboardLayout>
            );
          }} />
          <Route path="/assessments/:assessmentId" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <AssessmentDetail />
              </DashboardLayout>
            );
          }} />
          <Route path="/assessments/:assessmentId/take" component={AssessmentTaking} />
          <Route path="/assessments/:assessmentId/attempt/:attemptId" component={AssessmentTaking} />
          <Route path="/examination-center" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <ExaminationCenter />
              </DashboardLayout>
            );
          }} />
          <Route path="/examinations/:sessionId/take" component={ExaminationTaking} />
          <Route path="/examinations/:sessionId/results" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <ExaminationResults />
              </DashboardLayout>
            );
          }} />
          <Route path="/certificates" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            return (
              <DashboardLayout userType={userType}>
                <CertificateList />
              </DashboardLayout>
            );
          }} />
          <Route path="/accreditation" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'system_admin') {
              return <NotFound />;
            }
            return <AccreditationManagement />;
          }} />
          <Route path="/corporate-accounts" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'corporate_admin') {
              return <NotFound />;
            }
            return <CorporateAccountManagement />;
          }} />
          <Route path="/question-bank" component={() => {
            const userToCheck = user || currentUser;
            if (!['system_admin', 'partner_admin'].includes(userToCheck?.userType)) {
              return <NotFound />;
            }
            return <QuestionBankManagement />;
          }} />
          <Route path="/system-setup" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'system_admin') {
              return <NotFound />;
            }
            return <SystemSetup />;
          }} />
          <Route path="/examinations" component={() => {
            const userToCheck = user || currentUser;
            const userType = userToCheck?.userType === 'corporate_admin' ? 'corporate' : 
                            userToCheck?.userType === 'partner_admin' ? 'partner' :
                            userToCheck?.userType === 'system_admin' ? 'admin' : 'learner';
            
            // System admin gets the examinations management page
            if (userToCheck?.userType === 'system_admin') {
              return <Examinations />;
            }
            
            // Other users get the examination center page
            return (
              <DashboardLayout userType={userType}>
                <ExaminationCenter />
              </DashboardLayout>
            );
          }} />
          <Route path="/elearning" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'system_admin') {
              return <NotFound />;
            }
            return <ELearning />;
          }} />
          <Route path="/ebooks" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'system_admin') {
              return <NotFound />;
            }
            return <EBooks />;
          }} />
          <Route path="/qa" component={() => {
            const userToCheck = user || currentUser;
            if (userToCheck?.userType !== 'qa_admin') {
              return <NotFound />;
            }
            return <QAAccreditation />;
          }} />
          <Route path="/cms" component={() => {
            const userToCheck = user || currentUser;
            if (!['system_admin', 'website_admin'].includes(userToCheck?.userType || '')) {
              return <NotFound />;
            }
            return <CMS />;
          }} />

        </>
      )}
      <Route component={NotFound} />
    </Switch>
  );
}

function App() {
  return (
    <QueryClientProvider client={queryClient}>
      <TooltipProvider>
        <Toaster />
        <Router />
      </TooltipProvider>
    </QueryClientProvider>
  );
}

export default App;
